--[[----------------------------------------------------------------------------
    PhotoContextAPI.lua

    HTTP client for communicating with OpenRouter API using vision models.
    Handles image encoding, location analysis, and geocoding.
------------------------------------------------------------------------------]]

local LrHttp = import "LrHttp"
local LrFileUtils = import "LrFileUtils"
local LrPathUtils = import "LrPathUtils"
local LrStringUtils = import "LrStringUtils"
local LrErrors = import "LrErrors"

local PhotoContextAPI = {}

-- Configuration
PhotoContextAPI.API_URL = "https://openrouter.ai/api/v1/chat/completions"
PhotoContextAPI.ANALYZE_TIMEOUT = 120000  -- 120 seconds
PhotoContextAPI.DEFAULT_MODEL = "openai/gpt-4o-mini"

-- Available models with vision capability
-- Prices are approximate cost per photo analysis
PhotoContextAPI.VISION_MODELS = {
    -- Paid models (reliable, better accuracy)
    { id = "openai/gpt-4o-mini", name = "GPT-4o Mini (~$0.001/photo) - Recommended" },
    { id = "google/gemini-2.0-flash-001", name = "Gemini 2.0 Flash (~$0.001/photo)" },
    { id = "openai/gpt-4o", name = "GPT-4o (~$0.01/photo)" },
    { id = "anthropic/claude-sonnet-4", name = "Claude Sonnet 4 (~$0.01/photo)" },
    -- Free models (may have rate limits)
    { id = "qwen/qwen-2.5-vl-7b-instruct:free", name = "Qwen 2.5 VL 7B (Free)" },
    { id = "google/gemini-2.0-flash-exp:free", name = "Gemini 2.0 Flash Exp (Free)" },
}

--------------------------------------------------------------------------------
-- Base64 Encoding
-- Pure Lua implementation for Lightroom SDK compatibility
--------------------------------------------------------------------------------

local b64chars = 'ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/'

local function base64Encode(data)
    return ((data:gsub('.', function(x)
        local r, b = '', x:byte()
        for i = 8, 1, -1 do
            r = r .. (b % 2^i - b % 2^(i-1) > 0 and '1' or '0')
        end
        return r
    end) .. '0000'):gsub('%d%d%d?%d?%d?%d?', function(x)
        if #x < 6 then return '' end
        local c = 0
        for i = 1, 6 do
            c = c + (x:sub(i, i) == '1' and 2^(6-i) or 0)
        end
        return b64chars:sub(c+1, c+1)
    end) .. ({ '', '==', '=' })[#data % 3 + 1])
end

PhotoContextAPI.base64Encode = base64Encode

--------------------------------------------------------------------------------
-- JSON Building Utilities
-- Lightroom SDK doesn't include a JSON library
--------------------------------------------------------------------------------

local function escapeJsonString(str)
    if not str then return "" end
    str = str:gsub('\\', '\\\\')
    str = str:gsub('"', '\\"')
    str = str:gsub('\n', '\\n')
    str = str:gsub('\r', '\\r')
    str = str:gsub('\t', '\\t')
    return str
end

--------------------------------------------------------------------------------
-- JSON Parsing Utilities
--------------------------------------------------------------------------------

local function parseJsonString(json, key)
    if not json then return nil end

    -- Find "key":
    local keyPattern = '"' .. key .. '"%s*:%s*'
    local keyStart, keyEnd = json:find(keyPattern)
    if not keyStart then return nil end

    -- Check what comes after the colon
    local afterColon = json:sub(keyEnd + 1, keyEnd + 1)

    -- If it's a quote, extract the string value
    if afterColon == '"' then
        local result = ""
        local i = keyEnd + 2
        while i <= #json do
            local char = json:sub(i, i)
            if char == '\\' and i < #json then
                local nextChar = json:sub(i + 1, i + 1)
                if nextChar == '"' then
                    result = result .. '"'
                    i = i + 2
                elseif nextChar == 'n' then
                    result = result .. '\n'
                    i = i + 2
                elseif nextChar == 'r' then
                    result = result .. '\r'
                    i = i + 2
                elseif nextChar == 't' then
                    result = result .. '\t'
                    i = i + 2
                elseif nextChar == '\\' then
                    result = result .. '\\'
                    i = i + 2
                else
                    result = result .. char
                    i = i + 1
                end
            elseif char == '"' then
                break
            else
                result = result .. char
                i = i + 1
            end
        end
        return result
    end

    return nil
end

local function parseJsonNumber(json, key)
    local pattern = '"' .. key .. '"%s*:%s*([%d%.%-]+)'
    local numStr = string.match(json, pattern)
    if numStr then
        return tonumber(numStr)
    end
    return nil
end

local function parseJsonBoolean(json, key)
    local pattern = '"' .. key .. '"%s*:%s*(true)'
    return string.match(json, pattern) ~= nil
end

-- Extract content from OpenRouter response
local function extractContent(response)
    if not response then return nil end

    -- Response format: {"choices":[{"message":{"content":"..."}}]}
    -- Find "content": and then extract the string value
    local contentStart = response:find('"content"%s*:%s*"')
    if not contentStart then
        -- Try alternate format where content might be after message
        contentStart = response:find('"message"%s*:%s*{[^}]*"content"%s*:%s*"')
        if contentStart then
            contentStart = response:find('"content"%s*:%s*"', contentStart)
        end
    end

    if not contentStart then
        return nil
    end

    -- Find the opening quote of the content value
    local valueStart = response:find('"', contentStart + 9)
    if not valueStart then return nil end

    -- Extract content by finding the closing quote (handling escaped quotes)
    local content = ""
    local i = valueStart + 1
    while i <= #response do
        local char = response:sub(i, i)
        if char == '\\' and i < #response then
            local nextChar = response:sub(i + 1, i + 1)
            if nextChar == '"' then
                content = content .. '"'
                i = i + 2
            elseif nextChar == 'n' then
                content = content .. '\n'
                i = i + 2
            elseif nextChar == 'r' then
                content = content .. '\r'
                i = i + 2
            elseif nextChar == 't' then
                content = content .. '\t'
                i = i + 2
            elseif nextChar == '\\' then
                content = content .. '\\'
                i = i + 2
            else
                content = content .. char
                i = i + 1
            end
        elseif char == '"' then
            -- End of string
            break
        else
            content = content .. char
            i = i + 1
        end
    end

    return content
end

-- Extract JSON object from text (finds first { ... } block)
local function extractJsonObject(text)
    if not text then return nil end

    -- First try markdown code blocks
    local jsonFromBlock = text:match("```json%s*(%b{})%s*```")
        or text:match("```%s*(%b{})%s*```")
    if jsonFromBlock then
        return jsonFromBlock
    end

    -- Find the first { and match to closing }
    local startPos = text:find("{")
    if not startPos then return nil end

    -- Use balanced matching to find the complete JSON object
    local depth = 0
    local inString = false
    local escape = false

    for i = startPos, #text do
        local char = text:sub(i, i)

        if escape then
            escape = false
        elseif char == "\\" and inString then
            escape = true
        elseif char == '"' and not escape then
            inString = not inString
        elseif not inString then
            if char == "{" then
                depth = depth + 1
            elseif char == "}" then
                depth = depth - 1
                if depth == 0 then
                    return text:sub(startPos, i)
                end
            end
        end
    end

    return nil
end

-- Parse the structured JSON response from the model
local function parseLocationResponse(content)
    if not content then
        return nil, "No content in response"
    end

    -- Extract JSON object from the response
    local jsonStr = extractJsonObject(content)

    -- If no JSON object found, try the whole content
    if not jsonStr then
        jsonStr = content
    end

    local result = {
        sublocation = parseJsonString(jsonStr, "sublocation"),
        street_address = parseJsonString(jsonStr, "street_address"),
        city = parseJsonString(jsonStr, "city"),
        state_province = parseJsonString(jsonStr, "state_province"),
        country = parseJsonString(jsonStr, "country"),
        latitude = parseJsonNumber(jsonStr, "latitude"),
        longitude = parseJsonNumber(jsonStr, "longitude"),
        confidence = parseJsonNumber(jsonStr, "confidence"),
        explanation = parseJsonString(jsonStr, "explanation"),
        caption = parseJsonString(jsonStr, "caption"),
        raw_model_output = content,
    }

    -- Treat empty strings as nil
    if result.sublocation == "" then result.sublocation = nil end
    if result.street_address == "" then result.street_address = nil end
    if result.city == "" then result.city = nil end
    if result.state_province == "" then result.state_province = nil end
    if result.country == "" then result.country = nil end
    if result.caption == "" then result.caption = nil end

    return result, nil
end

--------------------------------------------------------------------------------
-- URL Encoding for geocoding queries
--------------------------------------------------------------------------------

local function urlEncode(str)
    if str then
        str = string.gsub(str, "\n", "\r\n")
        str = string.gsub(str, "([^%w %-%_%.%~])",
            function(c) return string.format("%%%02X", string.byte(c)) end)
        str = string.gsub(str, " ", "+")
    end
    return str
end

--------------------------------------------------------------------------------
-- Geocoding using Nominatim (OpenStreetMap)
--------------------------------------------------------------------------------

local NOMINATIM_URL = "https://nominatim.openstreetmap.org/search"
local GEOCODE_TIMEOUT = 10000

function PhotoContextAPI.geocodeLocation(locationString)
    if not locationString or locationString == "" then
        return nil, nil, "No location provided"
    end

    local url = NOMINATIM_URL .. "?q=" .. urlEncode(locationString) .. "&format=json&limit=1"

    local headers = {
        { field = "User-Agent", value = "PhotoContext/1.0" },
    }

    local response, responseHeaders = LrHttp.get(url, headers, GEOCODE_TIMEOUT)

    if response then
        local lat = string.match(response, '"lat"%s*:%s*"([%-]?[%d%.]+)"')
        local lon = string.match(response, '"lon"%s*:%s*"([%-]?[%d%.]+)"')

        if lat and lon then
            local latNum = tonumber(lat)
            local lonNum = tonumber(lon)
            if latNum and lonNum then
                return latNum, lonNum, nil
            end
        end

        return nil, nil, "Could not parse coordinates from response"
    else
        return nil, nil, "Geocoding request failed"
    end
end

function PhotoContextAPI.geocodeAddress(city, state, country)
    local parts = {}
    if city and city ~= "" then table.insert(parts, city) end
    if state and state ~= "" then table.insert(parts, state) end
    if country and country ~= "" then table.insert(parts, country) end

    if #parts == 0 then
        return nil, nil, "No address components provided"
    end

    local locationString = table.concat(parts, ", ")
    return PhotoContextAPI.geocodeLocation(locationString)
end

--------------------------------------------------------------------------------
-- Image Processing
--------------------------------------------------------------------------------

local function getContentType(filePath)
    local ext = LrPathUtils.extension(filePath)
    if ext then
        ext = ext:lower()
        if ext == "jpg" or ext == "jpeg" then
            return "image/jpeg"
        elseif ext == "png" then
            return "image/png"
        elseif ext == "gif" then
            return "image/gif"
        elseif ext == "webp" then
            return "image/webp"
        end
    end
    return "image/jpeg"  -- Default to JPEG
end

local function loadImageAsBase64(filePath)
    local content = LrFileUtils.readFile(filePath)
    if not content then
        return nil, "Could not read file: " .. filePath
    end
    return base64Encode(content), nil
end

--------------------------------------------------------------------------------
-- User Prompt for Location Analysis (embedded in user message for compatibility)
--------------------------------------------------------------------------------

local USER_PROMPT_TEMPLATE = [[Analyze this image and identify the geographic location where it was taken.

Look for visual clues such as landmarks, signage, architecture, natural features, and cultural elements.

%s

Respond ONLY with a JSON object in this exact format:
{"sublocation":"specific place or landmark","street_address":"street if visible","city":"city name","state_province":"state or region","country":"country name","latitude":0.0,"longitude":0.0,"confidence":0.0,"explanation":"how you identified this","caption":"photo description"}

Use "" for unknown text fields, null for unknown coordinates. Confidence: 0.0-1.0]]

--------------------------------------------------------------------------------
-- Main API Method
--------------------------------------------------------------------------------

function PhotoContextAPI.analyzePhoto(apiKey, photoPath, hint, model)
    if not apiKey or apiKey == "" then
        return nil, "OpenRouter API key is not configured. Please set it in Plugin Manager."
    end

    -- Use default model if not specified
    model = model or PhotoContextAPI.DEFAULT_MODEL

    -- Verify file exists
    if not LrFileUtils.exists(photoPath) then
        return nil, "File not found: " .. photoPath
    end

    -- Check file extension
    local ext = LrPathUtils.extension(photoPath)
    if ext then
        ext = ext:lower()
        if ext ~= "jpg" and ext ~= "jpeg" and ext ~= "png" and ext ~= "gif" and ext ~= "webp" then
            return nil, "Unsupported image format. Please use JPG, PNG, GIF, or WebP."
        end
    end

    -- Load and encode image
    local base64Image, err = loadImageAsBase64(photoPath)
    if not base64Image then
        return nil, err
    end

    local contentType = getContentType(photoPath)

    -- Build user message with optional hint
    local hintText = ""
    if hint and hint ~= "" then
        hintText = "User hint: " .. hint
    end
    local userMessage = string.format(USER_PROMPT_TEMPLATE, hintText)

    -- Build the request body (no system message for better vision model compatibility)
    local requestBody = string.format([[{
    "model": "%s",
    "messages": [
        {
            "role": "user",
            "content": [
                {
                    "type": "text",
                    "text": "%s"
                },
                {
                    "type": "image_url",
                    "image_url": {
                        "url": "data:%s;base64,%s"
                    }
                }
            ]
        }
    ],
    "max_tokens": 1024
}]],
        escapeJsonString(model),
        escapeJsonString(userMessage),
        contentType,
        base64Image
    )

    -- Set up headers
    local headers = {
        { field = "Authorization", value = "Bearer " .. apiKey },
        { field = "Content-Type", value = "application/json" },
        { field = "HTTP-Referer", value = "https://github.com/bmaestrini/photocontext" },
        { field = "X-Title", value = "PhotoContext" },
    }

    -- Make the API request
    local response, responseHeaders = LrHttp.post(
        PhotoContextAPI.API_URL,
        requestBody,
        headers,
        "POST",
        PhotoContextAPI.ANALYZE_TIMEOUT
    )

    if response then
        local status = responseHeaders and responseHeaders.status

        if status == 200 then
            -- Extract the content from the response
            local content = extractContent(response)
            if content then
                return parseLocationResponse(content)
            else
                return nil, "Could not parse model response"
            end

        elseif status == 401 then
            return nil, "Invalid API key. Please check your OpenRouter API key in Plugin Manager."

        elseif status == 402 then
            return nil, "Insufficient credits. Please add credits to your OpenRouter account."

        elseif status == 429 then
            return nil, "Rate limit exceeded. Please wait a moment and try again."

        elseif status == 500 or status == 502 or status == 503 then
            return nil, "OpenRouter server error. Please try again later."

        else
            local errorMsg = parseJsonString(response, "error")
                or parseJsonString(response, "message")
                or response:sub(1, 200)
            return nil, "API error (status " .. tostring(status) .. "): " .. tostring(errorMsg)
        end
    else
        local errMsg = "Request failed - could not connect to OpenRouter"
        if responseHeaders and responseHeaders.error then
            if type(responseHeaders.error) == "table" then
                errMsg = errMsg .. " | " .. (responseHeaders.error.errorCode or "Unknown error")
            else
                errMsg = errMsg .. " | " .. tostring(responseHeaders.error)
            end
        end
        return nil, errMsg
    end
end

-- Analyze multiple photos with progress callback
function PhotoContextAPI.analyzePhotos(apiKey, photoPaths, progressCallback, model)
    local results = {
        success = {},
        failed = {},
        total = #photoPaths,
    }

    for i, photoPath in ipairs(photoPaths) do
        if progressCallback then
            local shouldContinue = progressCallback(i, #photoPaths, photoPath)
            if shouldContinue == false then
                break
            end
        end

        local result, err = PhotoContextAPI.analyzePhoto(apiKey, photoPath, nil, model)

        if result then
            table.insert(results.success, {
                path = photoPath,
                result = result,
            })
        else
            table.insert(results.failed, {
                path = photoPath,
                error = err,
            })
        end
    end

    return results
end

return PhotoContextAPI
