--[[----------------------------------------------------------------------------
    PhotoContextDialog.lua

    UI dialogs for displaying analysis results and allowing user confirmation.
------------------------------------------------------------------------------]]

local LrView = import "LrView"
local LrDialogs = import "LrDialogs"
local LrBinding = import "LrBinding"
local LrColor = import "LrColor"
local LrApplication = import "LrApplication"
local LrTasks = import "LrTasks"
local LrFunctionContext = import "LrFunctionContext"
local LrProgressScope = import "LrProgressScope"
local LrPathUtils = import "LrPathUtils"
local LrPrefs = import "LrPrefs"

local PhotoContextAPI = require "PhotoContextAPI"

local PhotoContextDialog = {}

--------------------------------------------------------------------------------
-- Preferences Management
--------------------------------------------------------------------------------

local function getPrefs()
    local prefs = LrPrefs.prefsForPlugin()
    if not prefs.apiKey then
        prefs.apiKey = ""
    end
    if not prefs.selectedModel then
        prefs.selectedModel = PhotoContextAPI.DEFAULT_MODEL
    end
    if prefs.confidenceThreshold == nil then
        prefs.confidenceThreshold = 0.7
    end
    if prefs.autoApplyHighConfidence == nil then
        prefs.autoApplyHighConfidence = false
    end
    return prefs
end

PhotoContextDialog.getPrefs = getPrefs

--------------------------------------------------------------------------------
-- Result Display Dialog
--------------------------------------------------------------------------------

local function formatConfidence(confidence)
    if not confidence then
        return "Unknown"
    end
    local percent = math.floor(confidence * 100 + 0.5)
    local level = "Low"
    if confidence >= 0.7 then
        level = "High"
    elseif confidence >= 0.4 then
        level = "Medium"
    end
    return string.format("%d%% (%s)", percent, level)
end

local function formatCoordinates(lat, lon)
    if lat and lon then
        return string.format("%.6f, %.6f", lat, lon)
    end
    return "Not detected"
end

-- Build a display string from structured location fields
local function buildLocationDisplay(result)
    local parts = {}
    if result.sublocation then table.insert(parts, result.sublocation) end
    if result.street_address then table.insert(parts, result.street_address) end
    if result.city then table.insert(parts, result.city) end
    if result.state_province then table.insert(parts, result.state_province) end
    if result.country then table.insert(parts, result.country) end

    if #parts > 0 then
        return table.concat(parts, ", ")
    end
    return "Unknown"
end

-- Show result for a single photo and let user decide what to apply
function PhotoContextDialog.showSingleResult(photo, result)
    return LrFunctionContext.callWithContext("showSingleResult", function(context)
        local props = LrBinding.makePropertyTable(context)

        -- Initialize properties from structured API response
        props.guessedLocation = buildLocationDisplay(result)
        props.confidence = result.confidence or 0
        props.confidenceDisplay = formatConfidence(result.confidence)
        props.explanation = result.explanation or "No explanation provided"
        props.coordinates = formatCoordinates(result.latitude, result.longitude)
        props.hasCoordinates = (result.latitude ~= nil and result.longitude ~= nil)
        props.latitude = result.latitude
        props.longitude = result.longitude

        -- User choices
        props.applyGPS = props.hasCoordinates
        props.applyCaption = true
        props.applyLocation = true
        props.applyStreetAddress = (result.street_address ~= nil)
        props.applyCity = true
        props.applyState = true
        props.applyCountry = true

        -- Use structured fields directly from API response
        -- Use explanation as caption (describes the identified location)
        props.captionText = result.explanation or result.caption or ""
        props.locationText = result.sublocation or ""
        props.streetAddressText = result.street_address or ""
        props.cityText = result.city or ""
        props.stateText = result.state_province or ""
        props.countryText = result.country or ""

        -- Debug: raw model output
        props.showRawOutput = false
        props.rawOutput = result.raw_model_output or "(no output)"

        local photoName = LrPathUtils.leafName(photo:getRawMetadata("path") or "Unknown")

        local f = LrView.osFactory()

        local contents = f:column {
            bind_to_object = props,
            spacing = f:control_spacing(),

            f:row {
                f:static_text {
                    title = "Photo: " .. photoName,
                    font = "<system/bold>",
                },
            },

            f:separator { fill_horizontal = 1 },

            -- AI Result Section
            f:group_box {
                title = "AI Analysis Result",
                fill_horizontal = 1,

                f:row {
                    f:static_text { title = "Location:", width = 100, alignment = "right" },
                    f:static_text {
                        title = LrView.bind("guessedLocation"),
                        width_in_chars = 40,
                        height_in_lines = 2,
                        wraps = true,
                    },
                },

                f:row {
                    f:static_text { title = "Confidence:", width = 100, alignment = "right" },
                    f:static_text {
                        title = LrView.bind("confidenceDisplay"),
                        text_color = LrView.bind {
                            key = "confidence",
                            transform = function(value)
                                if value >= 0.7 then
                                    return LrColor(0, 0.6, 0)
                                elseif value >= 0.4 then
                                    return LrColor(0.8, 0.6, 0)
                                else
                                    return LrColor(0.8, 0, 0)
                                end
                            end,
                        },
                    },
                },

                f:row {
                    f:static_text { title = "Coordinates:", width = 100, alignment = "right" },
                    f:static_text {
                        title = LrView.bind("coordinates"),
                    },
                },

                f:row {
                    f:static_text { title = "Explanation:", width = 100, alignment = "right" },
                    f:static_text {
                        title = LrView.bind("explanation"),
                        width_in_chars = 40,
                        height_in_lines = 4,
                        wraps = true,
                    },
                },

                -- Show raw output for debugging if parsing failed
                f:row {
                    visible = LrView.bind {
                        key = "showRawOutput",
                        transform = function(value) return value == true end,
                    },
                    f:static_text { title = "Raw Output:", width = 100, alignment = "right" },
                    f:edit_field {
                        value = LrView.bind("rawOutput"),
                        width_in_chars = 40,
                        height_in_lines = 4,
                        enabled = false,
                    },
                },

                f:row {
                    f:static_text { title = "", width = 100 },
                    f:push_button {
                        title = LrView.bind {
                            key = "showRawOutput",
                            transform = function(value)
                                return value and "Hide Raw Output" or "Show Raw Output"
                            end,
                        },
                        action = function()
                            props.showRawOutput = not props.showRawOutput
                        end,
                    },
                },
            },

            f:separator { fill_horizontal = 1 },

            -- Metadata to Apply Section
            f:group_box {
                title = "Metadata to Apply",
                fill_horizontal = 1,

                f:row {
                    f:checkbox {
                        title = "GPS Coordinates:",
                        value = LrView.bind("applyGPS"),
                        enabled = LrView.bind("hasCoordinates"),
                    },
                    f:edit_field {
                        value = LrView.bind("coordinates"),
                        width_in_chars = 25,
                        enabled = false,
                    },
                    f:push_button {
                        title = "Lookup",
                        action = function()
                            LrTasks.startAsyncTask(function()
                                -- Build location string from fields
                                local parts = {}
                                if props.cityText and props.cityText ~= "" then
                                    table.insert(parts, props.cityText)
                                end
                                if props.stateText and props.stateText ~= "" then
                                    table.insert(parts, props.stateText)
                                end
                                if props.countryText and props.countryText ~= "" then
                                    table.insert(parts, props.countryText)
                                end

                                if #parts == 0 then
                                    LrDialogs.message("Lookup", "No location to look up", "warning")
                                    return
                                end

                                local locationString = table.concat(parts, ", ")
                                props.coordinates = "Looking up..."

                                local lat, lon, err = PhotoContextAPI.geocodeLocation(locationString)
                                if lat and lon then
                                    props.latitude = lat
                                    props.longitude = lon
                                    props.coordinates = string.format("%.6f, %.6f", lat, lon)
                                    props.hasCoordinates = true
                                    props.applyGPS = true
                                else
                                    props.coordinates = "Not found"
                                    LrDialogs.message("Lookup Failed", "Could not find coordinates for:\n" .. locationString, "warning")
                                end
                            end)
                        end,
                    },
                },

                f:row {
                    f:checkbox {
                        title = "Caption:",
                        value = LrView.bind("applyCaption"),
                    },
                    f:edit_field {
                        value = LrView.bind("captionText"),
                        width_in_chars = 50,
                        height_in_lines = 3,
                    },
                },

                f:row {
                    f:checkbox {
                        title = "Location (IPTC):",
                        value = LrView.bind("applyLocation"),
                    },
                    f:edit_field {
                        value = LrView.bind("locationText"),
                        width_in_chars = 35,
                    },
                },

                f:row {
                    f:checkbox {
                        title = "Street Address:",
                        value = LrView.bind("applyStreetAddress"),
                    },
                    f:edit_field {
                        value = LrView.bind("streetAddressText"),
                        width_in_chars = 35,
                    },
                },

                f:row {
                    f:checkbox {
                        title = "City:",
                        value = LrView.bind("applyCity"),
                    },
                    f:edit_field {
                        value = LrView.bind("cityText"),
                        width_in_chars = 25,
                    },
                },

                f:row {
                    f:checkbox {
                        title = "State/Province:",
                        value = LrView.bind("applyState"),
                    },
                    f:edit_field {
                        value = LrView.bind("stateText"),
                        width_in_chars = 25,
                    },
                },

                f:row {
                    f:checkbox {
                        title = "Country:",
                        value = LrView.bind("applyCountry"),
                    },
                    f:edit_field {
                        value = LrView.bind("countryText"),
                        width_in_chars = 25,
                    },
                },
            },
        }

        local dialogResult = LrDialogs.presentModalDialog {
            title = "PhotoContext - Analysis Result",
            contents = contents,
            actionVerb = "Apply",
            cancelVerb = "Skip",
            otherVerb = "Cancel All",
        }

        if dialogResult == "ok" then
            -- Return what to apply
            return {
                apply = true,
                gps = props.applyGPS and props.hasCoordinates,
                latitude = props.latitude or result.latitude,
                longitude = props.longitude or result.longitude,
                caption = props.applyCaption and props.captionText ~= "" and props.captionText or nil,
                location = props.applyLocation and props.locationText ~= "" and props.locationText or nil,
                streetAddress = props.applyStreetAddress and props.streetAddressText ~= "" and props.streetAddressText or nil,
                city = props.applyCity and props.cityText ~= "" and props.cityText or nil,
                state = props.applyState and props.stateText ~= "" and props.stateText or nil,
                country = props.applyCountry and props.countryText ~= "" and props.countryText or nil,
                -- Store AI metadata
                aiGuessedLocation = props.guessedLocation,
                aiConfidence = tostring(result.confidence),
                aiExplanation = result.explanation,
            }
        elseif dialogResult == "other" then
            return { cancelAll = true }
        else
            return { apply = false }
        end
    end)
end

-- Show summary after batch processing
function PhotoContextDialog.showBatchSummary(successCount, failedCount, skippedCount, errors)
    local message = string.format(
        "Analysis Complete\n\nApplied: %d\nSkipped: %d\nFailed: %d",
        successCount, skippedCount, failedCount
    )

    if failedCount > 0 and #errors > 0 then
        message = message .. "\n\nErrors:\n"
        for i, err in ipairs(errors) do
            if i <= 5 then
                message = message .. "- " .. err .. "\n"
            end
        end
        if #errors > 5 then
            message = message .. "... and " .. (#errors - 5) .. " more"
        end
    end

    LrDialogs.message("PhotoContext", message, "info")
end

-- Show error dialog
function PhotoContextDialog.showError(title, message)
    LrDialogs.message(title, message, "critical")
end

-- Show info dialog
function PhotoContextDialog.showInfo(title, message)
    LrDialogs.message(title, message, "info")
end

-- Batch options dialog for multiple photos
function PhotoContextDialog.showBatchOptions(photoCount)
    return LrFunctionContext.callWithContext("showBatchOptions", function(context)
        local props = LrBinding.makePropertyTable(context)
        local prefs = getPrefs()

        props.useSharedHint = false
        props.sharedHint = ""
        props.autoApply = false
        props.selectedModel = prefs.selectedModel

        -- Build model items
        local modelItems = {}
        for _, model in ipairs(PhotoContextAPI.VISION_MODELS) do
            table.insert(modelItems, {
                title = model.name,
                value = model.id,
            })
        end

        local f = LrView.osFactory()

        local contents = f:column {
            bind_to_object = props,
            spacing = f:control_spacing(),

            f:row {
                f:static_text {
                    title = string.format("Analyze %d photos for location", photoCount),
                    font = "<system/bold>",
                },
            },

            f:separator { fill_horizontal = 1 },

            f:group_box {
                title = "Model Selection",
                fill_horizontal = 1,

                f:row {
                    f:static_text {
                        title = "Vision Model:",
                        width = 100,
                    },
                    f:popup_menu {
                        value = LrView.bind("selectedModel"),
                        items = modelItems,
                        width_in_chars = 30,
                    },
                },
            },

            f:group_box {
                title = "Batch Options",
                fill_horizontal = 1,

                f:row {
                    f:checkbox {
                        title = "Use same hint for all photos:",
                        value = LrView.bind("useSharedHint"),
                    },
                },

                f:row {
                    f:edit_field {
                        value = LrView.bind("sharedHint"),
                        width_in_chars = 50,
                        height_in_lines = 2,
                        enabled = LrView.bind("useSharedHint"),
                        placeholder_string = "e.g., Europe trip 2023, Beach vacation...",
                    },
                },

                f:separator { fill_horizontal = 1 },

                f:row {
                    f:checkbox {
                        title = "Auto-apply all results (no confirmation per photo)",
                        value = LrView.bind("autoApply"),
                    },
                },

                f:row {
                    f:static_text {
                        title = "When enabled, results will be applied automatically without review.",
                        text_color = LrColor(0.5, 0.5, 0.5),
                    },
                },
            },

            f:separator { fill_horizontal = 1 },

            f:row {
                f:static_text {
                    title = "Large batches may take several minutes. API costs apply per image.",
                    text_color = LrColor(0.5, 0.5, 0.5),
                },
            },
        }

        local dialogResult = LrDialogs.presentModalDialog {
            title = "PhotoContext - Batch Processing",
            contents = contents,
            actionVerb = "Start",
            cancelVerb = "Cancel",
        }

        if dialogResult == "ok" then
            -- Save the selected model for next time
            local prefs = LrPrefs.prefsForPlugin()
            prefs.selectedModel = props.selectedModel

            return {
                proceed = true,
                useSharedHint = props.useSharedHint,
                sharedHint = props.sharedHint,
                autoApply = props.autoApply,
                selectedModel = props.selectedModel,
            }
        else
            return { proceed = false }
        end
    end)
end

-- Ask for optional hint before analyzing a photo
function PhotoContextDialog.askForHint(photoName, currentModel)
    return LrFunctionContext.callWithContext("askForHint", function(context)
        local props = LrBinding.makePropertyTable(context)
        local prefs = getPrefs()

        props.hint = ""
        props.selectedModel = currentModel or prefs.selectedModel

        -- Build model items
        local modelItems = {}
        for _, model in ipairs(PhotoContextAPI.VISION_MODELS) do
            table.insert(modelItems, {
                title = model.name,
                value = model.id,
            })
        end

        local f = LrView.osFactory()

        local contents = f:column {
            bind_to_object = props,
            spacing = f:control_spacing(),

            f:row {
                f:static_text {
                    title = "Photo: " .. photoName,
                    font = "<system/bold>",
                },
            },

            f:row {
                f:static_text {
                    title = "Model:",
                    width = 50,
                },
                f:popup_menu {
                    value = LrView.bind("selectedModel"),
                    items = modelItems,
                    width_in_chars = 30,
                },
            },

            f:row {
                f:static_text {
                    title = "Provide optional hints to help the AI identify the location:",
                    height_in_lines = 2,
                    width_in_chars = 50,
                },
            },

            f:row {
                f:edit_field {
                    value = LrView.bind("hint"),
                    width_in_chars = 50,
                    height_in_lines = 3,
                    placeholder_string = "e.g., Wright Brothers first flight location, Outer Banks vacation 2019, near the lighthouse...",
                },
            },

            f:row {
                f:static_text {
                    title = "Leave empty to analyze without hints.",
                    text_color = LrColor(0.5, 0.5, 0.5),
                },
            },
        }

        local result = LrDialogs.presentModalDialog {
            title = "PhotoContext - Location Hints",
            contents = contents,
            actionVerb = "Analyze",
            cancelVerb = "Skip",
            otherVerb = "Cancel All",
        }

        if result == "ok" then
            -- Save the selected model for next time
            local prefs = LrPrefs.prefsForPlugin()
            prefs.selectedModel = props.selectedModel

            return {
                analyze = true,
                hint = props.hint,
                selectedModel = props.selectedModel,
            }
        elseif result == "other" then
            return { cancelAll = true }
        else
            return { analyze = false }
        end
    end)
end

return PhotoContextDialog
