--[[----------------------------------------------------------------------------
    PhotoContextLicense.lua

    License key validation for PhotoContext plugin.
    Uses simple hash verification compatible with Lua 5.1.
------------------------------------------------------------------------------]]

local LrPrefs = import "LrPrefs"
local LrDate = import "LrDate"
local LrStringUtils = import "LrStringUtils"

local PhotoContextLicense = {}

--------------------------------------------------------------------------------
-- Configuration
--------------------------------------------------------------------------------

local PRODUCT_CODE = "PHCX2025"
local VENDOR_KEY = "maestrini"
local SALT = "LRplugin"

--------------------------------------------------------------------------------
-- Simple Hash Functions (Lua 5.1 compatible)
--------------------------------------------------------------------------------

-- Simple string hash that produces consistent results
local function simpleHash(str)
    local hash = 5381
    for i = 1, #str do
        local c = string.byte(str, i)
        -- hash * 33 + c (using modulo to keep it bounded)
        hash = ((hash * 33) + c) % 4294967296
    end
    return hash
end

-- Convert number to hex string
local function toHex(num, digits)
    local hex = string.format("%x", num % (16 ^ digits))
    while #hex < digits do
        hex = "0" .. hex
    end
    return string.upper(hex)
end

-- Generate a hash from multiple inputs
local function hashString(str)
    local h1 = simpleHash(str)
    local h2 = simpleHash(str .. SALT)
    local h3 = simpleHash(SALT .. str)
    local combined = (h1 + h2 * 31 + h3 * 97) % 4294967296
    return toHex(combined, 8)
end

-- HMAC-like keyed hash
local function keyedHash(key, message)
    local inner = hashString(key .. message .. SALT)
    local outer = hashString(SALT .. inner .. key)
    return outer
end

--------------------------------------------------------------------------------
-- License Key Format: XXXX-XXXX-XXXX-XXXX-XXXX
-- Structure: [EMAIL_HASH]-[DATE]-[TYPE]-[CHECK1]-[CHECK2]
--------------------------------------------------------------------------------

local function normalizeEmail(email)
    if not email then return "" end
    return string.lower(LrStringUtils.trimWhitespace(email))
end

local function generateChecksum(email, dateCode, licenseType)
    local data = PRODUCT_CODE .. "|" .. email .. "|" .. dateCode .. "|" .. licenseType
    local hash = keyedHash(VENDOR_KEY, data)
    return string.sub(hash, 1, 8)
end

local function getEmailHash(email)
    local hash = hashString(email .. SALT)
    return string.sub(hash, 1, 4)
end

local function parseKey(licenseKey)
    if not licenseKey then return nil end

    -- Remove spaces and dashes, uppercase
    local clean = string.upper(string.gsub(licenseKey, "[%s%-]", ""))

    -- Must be exactly 20 characters (5 groups of 4)
    if #clean ~= 20 then return nil end

    return {
        emailHash = string.sub(clean, 1, 4),
        dateCode = string.sub(clean, 5, 8),
        licenseType = string.sub(clean, 9, 12),
        check1 = string.sub(clean, 13, 16),
        check2 = string.sub(clean, 17, 20),
    }
end

--------------------------------------------------------------------------------
-- Public API
--------------------------------------------------------------------------------

function PhotoContextLicense.validateKey(licenseKey, email)
    local parts = parseKey(licenseKey)
    if not parts then
        return false, "Invalid key format"
    end

    local normalEmail = normalizeEmail(email)
    if normalEmail == "" then
        return false, "Email required"
    end

    -- Verify email hash matches
    local emailHash = getEmailHash(normalEmail)
    if emailHash ~= parts.emailHash then
        return false, "Key does not match this email"
    end

    -- Verify checksum
    local expectedCheck = generateChecksum(normalEmail, parts.dateCode, parts.licenseType)
    local providedCheck = parts.check1 .. parts.check2

    if expectedCheck ~= providedCheck then
        return false, "Invalid license key"
    end

    -- Check expiration (dateCode is YYMM, "LIFE" for lifetime)
    if parts.dateCode ~= "LIFE" then
        local year = 2000 + tonumber(string.sub(parts.dateCode, 1, 2))
        local month = tonumber(string.sub(parts.dateCode, 3, 4))

        if year and month then
            local now = LrDate.currentTime()
            local currentYear = tonumber(LrDate.timeToUserFormat(now, "%Y"))
            local currentMonth = tonumber(LrDate.timeToUserFormat(now, "%m"))

            if currentYear > year or (currentYear == year and currentMonth > month) then
                return false, "License has expired"
            end
        end
    end

    -- License is valid
    return true, { type = "paid" }
end

function PhotoContextLicense.isLicensed()
    local prefs = LrPrefs.prefsForPlugin()
    local key = prefs.licenseKey
    local email = prefs.licenseEmail

    if not key or key == "" or not email or email == "" then
        return false, nil
    end

    local valid, info = PhotoContextLicense.validateKey(key, email)
    return valid, info
end

function PhotoContextLicense.saveLicense(licenseKey, email)
    local valid, info = PhotoContextLicense.validateKey(licenseKey, email)

    if valid then
        local prefs = LrPrefs.prefsForPlugin()
        prefs.licenseKey = licenseKey
        prefs.licenseEmail = email
        return true, info
    else
        return false, info
    end
end

function PhotoContextLicense.clearLicense()
    local prefs = LrPrefs.prefsForPlugin()
    prefs.licenseKey = nil
    prefs.licenseEmail = nil
end

function PhotoContextLicense.getLicenseInfo()
    local prefs = LrPrefs.prefsForPlugin()
    return {
        key = prefs.licenseKey or "",
        email = prefs.licenseEmail or "",
    }
end

-- Trial/demo mode limits
PhotoContextLicense.TRIAL_LIMIT = 5  -- photos per day without license

return PhotoContextLicense
