--[[----------------------------------------------------------------------------
    PhotoContextMenuItem.lua

    Main entry point for the "Analyze with PhotoContext" menu item.
    Handles photo selection, API communication, and metadata writing.

    Uses OpenRouter API directly without a backend server.
------------------------------------------------------------------------------]]

local LrApplication = import "LrApplication"
local LrTasks = import "LrTasks"
local LrFunctionContext = import "LrFunctionContext"
local LrProgressScope = import "LrProgressScope"
local LrDialogs = import "LrDialogs"
local LrPathUtils = import "LrPathUtils"
local LrFileUtils = import "LrFileUtils"
local LrExportSession = import "LrExportSession"

local LrPrefs = import "LrPrefs"

local PhotoContextAPI = require "PhotoContextAPI"
local PhotoContextDialog = require "PhotoContextDialog"
local PhotoContextLicense = require "PhotoContextLicense"

--------------------------------------------------------------------------------
-- Daily Trial Tracking
-- Resets each day, persists across plugin invocations
--------------------------------------------------------------------------------

local function getTodayKey()
    return os.date("%Y%m%d")
end

local function getSessionTrialCount()
    local prefs = LrPrefs.prefsForPlugin()
    local today = getTodayKey()

    -- Reset counter if it's a new day
    if prefs.trialDate ~= today then
        prefs.trialDate = today
        prefs.trialCount = 0
    end

    return prefs.trialCount or 0
end

local function incrementTrialCount(count)
    local prefs = LrPrefs.prefsForPlugin()
    local today = getTodayKey()

    -- Ensure we're on the right day
    if prefs.trialDate ~= today then
        prefs.trialDate = today
        prefs.trialCount = 0
    end

    prefs.trialCount = (prefs.trialCount or 0) + (count or 1)
    return prefs.trialCount
end

--------------------------------------------------------------------------------
-- Configuration
--------------------------------------------------------------------------------

local MAX_IMAGE_SIZE = 2048      -- Max dimension (pixels) for API submission
local MAX_FILE_SIZE = 1048576    -- 1MB - resize if larger
local JPEG_QUALITY = 0.80        -- Quality for resized exports

--------------------------------------------------------------------------------
-- Metadata Writing
--------------------------------------------------------------------------------

local function applyMetadataToPhoto(photo, metadata, catalog)
    catalog:withWriteAccessDo("Apply PhotoContext Location", function()
        -- Apply GPS coordinates if available
        if metadata.gps and metadata.latitude and metadata.longitude then
            photo:setRawMetadata("gps", {
                latitude = metadata.latitude,
                longitude = metadata.longitude,
            })
        end

        -- Apply caption
        if metadata.caption and metadata.caption ~= "" then
            photo:setRawMetadata("caption", metadata.caption)
        end

        -- Apply IPTC location fields
        local locationParts = {}
        if metadata.location and metadata.location ~= "" then
            table.insert(locationParts, metadata.location)
        end
        if metadata.streetAddress and metadata.streetAddress ~= "" then
            table.insert(locationParts, metadata.streetAddress)
        end
        if #locationParts > 0 then
            photo:setRawMetadata("location", table.concat(locationParts, ", "))
        end

        if metadata.city and metadata.city ~= "" then
            photo:setRawMetadata("city", metadata.city)
        end

        if metadata.state and metadata.state ~= "" then
            photo:setRawMetadata("stateProvince", metadata.state)
        end

        if metadata.country and metadata.country ~= "" then
            photo:setRawMetadata("country", metadata.country)
        end
    end)
end

--------------------------------------------------------------------------------
-- Image Resizing for API
--------------------------------------------------------------------------------

local function exportResizedPhoto(photo, catalog)
    -- Create temp directory path
    local tempDir = LrPathUtils.getStandardFilePath("temp")
    local tempFileName = "photocontext_" .. tostring(os.time()) .. "_" .. math.random(1000, 9999) .. ".jpg"
    local tempPath = LrPathUtils.child(tempDir, tempFileName)

    -- Export settings for a resized JPEG
    local exportSettings = {
        LR_export_destinationType = "specificFolder",
        LR_export_destinationPathPrefix = tempDir,
        LR_export_useSubfolder = false,

        LR_format = "JPEG",
        LR_jpeg_quality = JPEG_QUALITY,

        LR_size_doConstrain = true,
        LR_size_maxHeight = MAX_IMAGE_SIZE,
        LR_size_maxWidth = MAX_IMAGE_SIZE,
        LR_size_resizeType = "longEdge",
        LR_size_units = "pixels",

        LR_outputSharpeningOn = false,
        LR_metadata_keywordOptions = "flat",
        LR_removeLocationMetadata = true,
        LR_embeddedMetadataOption = "none",

        LR_collisionHandling = "rename",
        LR_tokens = "{{image_name}}",

        LR_includeFaceTagsAsKeywords = false,
        LR_includeFaceTagsInIptc = false,
    }

    local exportedPath = nil
    local exportError = nil

    -- Perform export
    local exportSession = LrExportSession({
        photosToExport = { photo },
        exportSettings = exportSettings,
    })

    for _, rendition in exportSession:renditions() do
        local success, pathOrMessage = rendition:waitForRender()
        if success then
            exportedPath = pathOrMessage
        else
            exportError = pathOrMessage or "Export failed"
        end
    end

    return exportedPath, exportError
end

--------------------------------------------------------------------------------
-- Photo Path Handling with Auto-Resize
--------------------------------------------------------------------------------

local function getPhotoPathForAnalysis(photo, catalog)
    local originalPath = photo:getRawMetadata("path")

    if not originalPath or not LrFileUtils.exists(originalPath) then
        return nil, false, "Could not access photo file"
    end

    local ext = LrPathUtils.extension(originalPath)
    if ext then
        ext = ext:lower()
    end

    -- Check if it's a supported format or needs export
    local needsExport = false
    local isSupported = (ext == "jpg" or ext == "jpeg" or ext == "png" or ext == "gif" or ext == "webp")

    if not isSupported then
        -- RAW or other format - must export
        needsExport = true
    else
        -- Check file size - resize if too large
        local fileAttrs = LrFileUtils.fileAttributes(originalPath)
        if fileAttrs and fileAttrs.fileSize and fileAttrs.fileSize > MAX_FILE_SIZE then
            needsExport = true
        end
    end

    if needsExport then
        -- Export a resized version
        local exportedPath, exportError = exportResizedPhoto(photo, catalog)
        if exportedPath then
            return exportedPath, true, nil  -- needsCleanup = true
        else
            return nil, false, "Failed to export resized image: " .. (exportError or "unknown error")
        end
    else
        -- Use original file
        return originalPath, false, nil
    end
end

--------------------------------------------------------------------------------
-- Main Processing Logic
--------------------------------------------------------------------------------

local function processPhotos()
    LrFunctionContext.callWithContext("PhotoContext.processPhotos", function(context)
        local catalog = LrApplication.activeCatalog()
        local photos = catalog:getTargetPhotos()

        if not photos or #photos == 0 then
            PhotoContextDialog.showError("PhotoContext", "No photos selected.")
            return
        end

        -- Get preferences
        local prefs = PhotoContextDialog.getPrefs()
        local apiKey = prefs.apiKey
        local selectedModel = prefs.selectedModel

        -- Check API key is configured
        if not apiKey or apiKey == "" then
            PhotoContextDialog.showError(
                "PhotoContext - Configuration Required",
                "OpenRouter API key is not configured.\n\n" ..
                "Please go to File > Plug-in Manager, select 'PhotoContext', " ..
                "and enter your API key.\n\n" ..
                "You can get an API key at openrouter.ai"
            )
            return
        end

        -- Check license
        local isLicensed, licenseInfo = PhotoContextLicense.isLicensed()
        local remainingTrial = PhotoContextLicense.TRIAL_LIMIT - getSessionTrialCount()

        if not isLicensed then
            if remainingTrial <= 0 then
                PhotoContextDialog.showError(
                    "PhotoContext - License Required",
                    "Trial limit reached (" .. PhotoContextLicense.TRIAL_LIMIT .. " photos per day).\n\n" ..
                    "Please purchase a license to continue using PhotoContext.\n\n" ..
                    "Go to File > Plug-in Manager > PhotoContext to enter your license key."
                )
                return
            elseif #photos > remainingTrial then
                local result = LrDialogs.confirm(
                    "PhotoContext - Trial Mode",
                    "You have " .. remainingTrial .. " trial photos remaining today.\n\n" ..
                    "You selected " .. #photos .. " photos. Only the first " .. remainingTrial .. " will be analyzed.\n\n" ..
                    "Purchase a license for unlimited use.",
                    "Continue",
                    "Cancel"
                )
                if result == "cancel" then
                    return
                end
                -- Limit photos to remaining trial
                local limitedPhotos = {}
                for i = 1, remainingTrial do
                    limitedPhotos[i] = photos[i]
                end
                photos = limitedPhotos
            else
                LrDialogs.message(
                    "PhotoContext - Trial Mode",
                    "Trial mode: " .. remainingTrial .. " photos remaining today.\n\n" ..
                    "Purchase a license for unlimited use.",
                    "info"
                )
            end
        end

        -- Batch options for multiple photos, or single photo flow
        local batchOptions = {
            proceed = true,
            useSharedHint = false,
            sharedHint = "",
            autoApply = false,
            selectedModel = selectedModel,
        }

        if #photos > 1 then
            batchOptions = PhotoContextDialog.showBatchOptions(#photos)
            if not batchOptions.proceed then
                return
            end
            selectedModel = batchOptions.selectedModel
        end

        -- Process photos
        local progressScope = LrProgressScope {
            title = "PhotoContext",
            functionContext = context,
        }

        local successCount = 0
        local failedCount = 0
        local skippedCount = 0
        local errors = {}

        for i, photo in ipairs(photos) do
            if progressScope:isCanceled() then
                break
            end

            local photoName = LrPathUtils.leafName(photo:getRawMetadata("path") or "Photo " .. i)
            progressScope:setCaption("Analyzing: " .. photoName)
            progressScope:setPortionComplete(i - 1, #photos)

            -- Get photo path (with auto-resize if needed)
            local photoPath, needsCleanup, pathError = getPhotoPathForAnalysis(photo, catalog)

            if not photoPath then
                failedCount = failedCount + 1
                table.insert(errors, photoName .. ": " .. (pathError or "Could not access photo file"))
            else
                -- Determine hint and model
                local hint = ""
                local shouldAnalyze = true
                local modelToUse = selectedModel

                if batchOptions.useSharedHint then
                    hint = batchOptions.sharedHint
                elseif not batchOptions.autoApply then
                    -- Ask for individual hint (single photo or non-auto batch)
                    local hintResult = PhotoContextDialog.askForHint(photoName, selectedModel)
                    if hintResult.cancelAll then
                        -- Clean up temp file before breaking
                        if needsCleanup and photoPath then
                            LrFileUtils.delete(photoPath)
                        end
                        break
                    elseif not hintResult.analyze then
                        skippedCount = skippedCount + 1
                        shouldAnalyze = false
                    else
                        hint = hintResult.hint
                        modelToUse = hintResult.selectedModel or selectedModel
                    end
                end

                if shouldAnalyze then
                    -- Analyze the photo with OpenRouter API
                    local result, err = PhotoContextAPI.analyzePhoto(apiKey, photoPath, hint, modelToUse)

                    if result then
                        -- Auto-geocode: Look up real coordinates from the identified location
                        -- Vision models often hallucinate GPS coordinates, so we use geocoding instead
                        local lat, lon = nil, nil

                        -- Try most specific first: sublocation + city + country
                        if result.sublocation and result.city then
                            local query = result.sublocation .. ", " .. result.city
                            if result.country then query = query .. ", " .. result.country end
                            lat, lon = PhotoContextAPI.geocodeLocation(query)
                        end

                        -- Try street + city + country
                        if not lat and result.street_address and result.city then
                            local query = result.street_address .. ", " .. result.city
                            if result.country then query = query .. ", " .. result.country end
                            lat, lon = PhotoContextAPI.geocodeLocation(query)
                        end

                        -- Fall back to city + state + country
                        if not lat and (result.city or result.state_province or result.country) then
                            lat, lon = PhotoContextAPI.geocodeAddress(
                                result.city,
                                result.state_province,
                                result.country
                            )
                        end

                        if lat and lon then
                            result.latitude = lat
                            result.longitude = lon
                            result.coordinates_source = "geocoded"
                        end
                        if batchOptions.autoApply then
                            -- Auto-apply: directly apply all metadata without confirmation
                            local metadata = {
                                apply = true,
                                gps = (result.latitude ~= nil and result.longitude ~= nil),
                                latitude = result.latitude,
                                longitude = result.longitude,
                                caption = result.explanation or result.caption,
                                location = result.sublocation,
                                streetAddress = result.street_address,
                                city = result.city,
                                state = result.state_province,
                                country = result.country,
                            }
                            applyMetadataToPhoto(photo, metadata, catalog)
                            successCount = successCount + 1
                        else
                            -- Show result dialog for user confirmation
                            local userChoice = PhotoContextDialog.showSingleResult(photo, result)

                            if userChoice.cancelAll then
                                -- Clean up temp file before breaking
                                if needsCleanup and photoPath then
                                    LrFileUtils.delete(photoPath)
                                end
                                break
                            elseif userChoice.apply then
                                applyMetadataToPhoto(photo, userChoice, catalog)
                                successCount = successCount + 1
                            else
                                skippedCount = skippedCount + 1
                            end
                        end
                    else
                        failedCount = failedCount + 1
                        table.insert(errors, photoName .. ": " .. (err or "Unknown error"))
                    end
                end

                -- Clean up temp file if needed
                if needsCleanup and photoPath then
                    LrFileUtils.delete(photoPath)
                end
            end
        end

        progressScope:done()

        -- Update trial count for unlicensed users
        if not isLicensed and successCount > 0 then
            incrementTrialCount(successCount)
        end

        -- Show summary
        if successCount > 0 or failedCount > 0 or skippedCount > 0 then
            PhotoContextDialog.showBatchSummary(successCount, failedCount, skippedCount, errors)
        end
    end)
end

--------------------------------------------------------------------------------
-- Entry Point
--------------------------------------------------------------------------------

LrTasks.startAsyncTask(function()
    processPhotos()
end)
