--[[----------------------------------------------------------------------------
    PluginInfoProvider.lua

    Plugin settings panel for PhotoContext - OpenRouter API configuration.
------------------------------------------------------------------------------]]

local LrView = import "LrView"
local LrPrefs = import "LrPrefs"
local LrTasks = import "LrTasks"
local LrDialogs = import "LrDialogs"
local LrColor = import "LrColor"
local LrHttp = import "LrHttp"
local LrBinding = import "LrBinding"

local PhotoContextAPI = require "PhotoContextAPI"
local PhotoContextLicense = require "PhotoContextLicense"

-- Build model selection items
local function getModelItems()
    local items = {}
    for _, model in ipairs(PhotoContextAPI.VISION_MODELS) do
        table.insert(items, {
            title = model.name,
            value = model.id,
        })
    end
    return items
end

return {
    startDialog = function(propertyTable)
        local prefs = LrPrefs.prefsForPlugin()

        -- Load saved values or defaults
        propertyTable.apiKey = prefs.apiKey or ""
        propertyTable.selectedModel = prefs.selectedModel or PhotoContextAPI.DEFAULT_MODEL

        -- License info
        local licenseInfo = PhotoContextLicense.getLicenseInfo()
        propertyTable.licenseKey = licenseInfo.key
        propertyTable.licenseEmail = licenseInfo.email

        -- Check license status
        local isLicensed, info = PhotoContextLicense.isLicensed()
        if isLicensed then
            propertyTable.licenseStatus = "Licensed (" .. (info.type or "valid") .. ")"
            propertyTable.licenseStatusColor = LrColor(0, 0.6, 0)
        else
            propertyTable.licenseStatus = "Not licensed - Trial mode (" .. PhotoContextLicense.TRIAL_LIMIT .. " photos/day)"
            propertyTable.licenseStatusColor = LrColor(0.8, 0.4, 0)
        end
    end,

    endDialog = function(propertyTable)
        local prefs = LrPrefs.prefsForPlugin()

        -- Save values to prefs
        prefs.apiKey = propertyTable.apiKey
        prefs.selectedModel = propertyTable.selectedModel
    end,

    sectionsForTopOfDialog = function(f, propertyTable)
        return {
            {
                title = "PhotoContext",
                bind_to_object = propertyTable,

                f:row {
                    f:static_text {
                        title = "Created by Bruno Maestrini",
                        font = "<system/bold>",
                    },
                },
                f:row {
                    f:static_text {
                        title = "Version 1.0",
                    },
                },
                f:row {
                    f:static_text {
                        title = "Uses AI vision models to analyze photos and determine their location.",
                        height_in_lines = 2,
                        width_in_chars = 50,
                    },
                },
            },
            {
                title = "OpenRouter API Settings",
                bind_to_object = propertyTable,

                f:row {
                    f:static_text {
                        title = "API Key:",
                        width = 80,
                    },
                    f:edit_field {
                        fill_horizontal = 1,
                        value = LrView.bind("apiKey"),
                        width_in_chars = 40,
                    },
                },
                f:row {
                    f:static_text {
                        title = "",
                        width = 80,
                    },
                    f:static_text {
                        title = "Get your API key at openrouter.ai (free to create)",
                        text_color = LrColor(0.5, 0.5, 0.5),
                    },
                },
                f:separator { fill_horizontal = 1 },
                f:row {
                    f:static_text {
                        title = "Model:",
                        width = 80,
                    },
                    f:popup_menu {
                        value = LrView.bind("selectedModel"),
                        items = getModelItems(),
                        width_in_chars = 35,
                    },
                },
                f:row {
                    f:static_text {
                        title = "",
                        width = 80,
                    },
                    f:static_text {
                        title = "GPT-4o Mini is recommended (~$0.001/photo). Free tier may have rate limits.",
                        text_color = LrColor(0.5, 0.5, 0.5),
                        height_in_lines = 2,
                        width_in_chars = 45,
                    },
                },
                f:separator { fill_horizontal = 1 },
                f:row {
                    f:static_text {
                        title = "",
                        width = 80,
                    },
                    f:push_button {
                        title = "Test API Connection",
                        action = function()
                            local apiKey = propertyTable.apiKey
                            if not apiKey or apiKey == "" then
                                LrDialogs.message("API Test", "Please enter your OpenRouter API key first.", "warning")
                                return
                            end

                            LrTasks.startAsyncTask(function()
                                local headers = {
                                    { field = "Authorization", value = "Bearer " .. apiKey },
                                    { field = "Content-Type", value = "application/json" },
                                }

                                local response, responseHeaders = LrHttp.get(
                                    "https://openrouter.ai/api/v1/models",
                                    headers,
                                    10000
                                )

                                if response then
                                    local status = responseHeaders and responseHeaders.status
                                    if status == 200 then
                                        -- Also save the key now since it's valid
                                        local prefs = LrPrefs.prefsForPlugin()
                                        prefs.apiKey = apiKey
                                        prefs.selectedModel = propertyTable.selectedModel
                                        LrDialogs.message("API Test", "Successfully connected to OpenRouter!\n\nYour API key is valid and has been saved.", "info")
                                    elseif status == 401 then
                                        LrDialogs.message("API Test", "Invalid API key.\n\nPlease check your OpenRouter API key.", "critical")
                                    else
                                        LrDialogs.message("API Test", "Unexpected response (status " .. tostring(status) .. ").\n\nPlease try again.", "warning")
                                    end
                                else
                                    LrDialogs.message("API Test", "Could not connect to OpenRouter.\n\nPlease check your internet connection.", "critical")
                                end
                            end)
                        end,
                    },
                },
            },
            {
                title = "License",
                bind_to_object = propertyTable,

                f:row {
                    f:static_text {
                        title = "Status:",
                        width = 80,
                    },
                    f:static_text {
                        title = LrView.bind("licenseStatus"),
                        text_color = LrView.bind("licenseStatusColor"),
                        font = "<system/bold>",
                    },
                },
                f:separator { fill_horizontal = 1 },
                f:row {
                    f:static_text {
                        title = "Email:",
                        width = 80,
                    },
                    f:edit_field {
                        value = LrView.bind("licenseEmail"),
                        width_in_chars = 30,
                    },
                },
                f:row {
                    f:static_text {
                        title = "License Key:",
                        width = 80,
                    },
                    f:edit_field {
                        value = LrView.bind("licenseKey"),
                        width_in_chars = 30,
                    },
                },
                f:row {
                    f:static_text {
                        title = "",
                        width = 80,
                    },
                    f:push_button {
                        title = "Activate License",
                        action = function()
                            local key = propertyTable.licenseKey
                            local email = propertyTable.licenseEmail

                            if not key or key == "" or not email or email == "" then
                                LrDialogs.message("License", "Please enter both email and license key.", "warning")
                                return
                            end

                            local success, info = PhotoContextLicense.saveLicense(key, email)
                            if success then
                                propertyTable.licenseStatus = "Licensed (" .. (info.type or "valid") .. ")"
                                propertyTable.licenseStatusColor = LrColor(0, 0.6, 0)
                                LrDialogs.message("License", "License activated successfully!\n\nThank you for your purchase.", "info")
                            else
                                LrDialogs.message("License", "Invalid license key.\n\n" .. tostring(info), "critical")
                            end
                        end,
                    },
                    f:push_button {
                        title = "Buy License",
                        action = function()
                            LrHttp.openUrlInBrowser("https://photocontext.bpix.es/")
                        end,
                    },
                },
            },
            {
                title = "Usage Notes",
                f:row {
                    f:static_text {
                        title = [[
- Supported formats: JPG, PNG, GIF, WebP (RAW auto-converted)
- GPT-4o Mini: ~$0.001/photo (1000 photos = ~$1)
- GPT-4o / Claude Sonnet 4: ~$0.01/photo (better accuracy)
- Free tier available but may have rate limits
- For best results, use images with visible landmarks]],
                        height_in_lines = 6,
                        width_in_chars = 55,
                    },
                },
            },
        }
    end,
}
